'use client'

import { useEffect } from 'react'
import { motion } from 'framer-motion'
import { AlertTriangle, RefreshCcw, Home } from 'lucide-react'
import Link from 'next/link'

export default function Error({
  error,
  reset,
}: {
  error: Error & { digest?: string }
  reset: () => void
}) {
  useEffect(() => {
    // Log error to console or error reporting service
    console.error('Application error:', error)
    
    // TODO: Send to error tracking service (e.g., Sentry)
    // if (process.env.NODE_ENV === 'production') {
    //   logErrorToService(error)
    // }
  }, [error])

  return (
    <div className="min-h-screen flex items-center justify-center bg-gray-950 px-4">
      <div className="max-w-2xl w-full">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
          className="text-center"
        >
          {/* Error Icon */}
          <motion.div
            initial={{ scale: 0 }}
            animate={{ scale: 1 }}
            transition={{ delay: 0.2, type: 'spring', stiffness: 200 }}
            className="flex justify-center mb-8"
          >
            <div className="relative">
              <div className="absolute inset-0 bg-gradient-to-r from-red-500 to-blue-600 rounded-full blur-2xl opacity-50 animate-pulse"></div>
              <div className="relative w-32 h-32 bg-gray-800/50 backdrop-blur-sm border-2 border-gray-700 rounded-full flex items-center justify-center">
                <AlertTriangle className="w-16 h-16 text-gradient" />
              </div>
            </div>
          </motion.div>

          {/* Error Message */}
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ delay: 0.3 }}
          >
            <h1 className="text-5xl md:text-6xl font-bold text-gradient mb-4">
              Oops!
            </h1>
            <h2 className="text-2xl md:text-3xl font-semibold text-white mb-4">
              Something went wrong
            </h2>
            <p className="text-lg text-gray-400 mb-8 max-w-lg mx-auto">
              We encountered an unexpected error. Don't worry, our team has been notified 
              and we're working to fix it.
            </p>

            {/* Error Details (Development Only) */}
            {process.env.NODE_ENV === 'development' && error.message && (
              <div className="mb-8 p-4 bg-red-500/10 border border-red-500/30 rounded-lg text-left max-w-lg mx-auto">
                <p className="text-sm text-red-400 font-mono break-all">
                  <strong>Error:</strong> {error.message}
                </p>
                {error.digest && (
                  <p className="text-xs text-red-400/70 font-mono mt-2">
                    Digest: {error.digest}
                  </p>
                )}
              </div>
            )}

            {/* Action Buttons */}
            <div className="flex flex-col sm:flex-row items-center justify-center gap-4">
              <motion.button
                whileHover={{ scale: 1.05 }}
                whileTap={{ scale: 0.95 }}
                onClick={reset}
                className="px-8 py-4 bg-gradient-to-r from-red-500 to-blue-600 text-white rounded-full font-semibold hover:shadow-glow-duo transition-all duration-300 flex items-center gap-2"
              >
                <RefreshCcw className="w-5 h-5" />
                Try Again
              </motion.button>

              <Link href="/">
                <motion.button
                  whileHover={{ scale: 1.05 }}
                  whileTap={{ scale: 0.95 }}
                  className="px-8 py-4 border border-gray-700 text-white rounded-full font-semibold hover:border-gray-600 hover:bg-gray-800/50 transition-all duration-300 flex items-center gap-2"
                >
                  <Home className="w-5 h-5" />
                  Go Home
                </motion.button>
              </Link>
            </div>
          </motion.div>

          {/* Decorative Elements */}
          <div className="absolute inset-0 overflow-hidden pointer-events-none">
            <div className="absolute w-96 h-96 -top-48 -left-48 bg-red-500/10 rounded-full blur-3xl"></div>
            <div className="absolute w-96 h-96 -bottom-48 -right-48 bg-blue-600/10 rounded-full blur-3xl"></div>
          </div>
        </motion.div>
      </div>
    </div>
  )
}

