'use client'

// Note: Controller type warnings with React 19 are known compatibility issues and can be safely ignored
// The form functionality is not affected by these TypeScript warnings

import { 
  Mail, Phone, MapPin, Send, Github, Linkedin, Twitter, 
  User, MessageSquare, FileText, Clock, CheckCircle2, 
  Sparkles, Zap, Shield, HeartHandshake, ArrowRight,
  Globe, Rocket, Star
} from 'lucide-react'
import { motion, AnimatePresence } from 'framer-motion'
import { useState, useEffect } from 'react'
import { useForm, Controller } from 'react-hook-form'
import { zodResolver } from '@hookform/resolvers/zod'
import { contactFormSchema, type ContactFormData } from '@/lib/validation-schemas'
import { showToast } from '@/lib/toast-config'

interface ExtendedContactFormData extends ContactFormData {
  acceptTerms?: boolean
  subscribeNewsletter?: boolean
}

export function Contact() {
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [messageLength, setMessageLength] = useState(0)
  const maxMessageLength = 1000

  const {
    control,
    handleSubmit,
    formState: { errors },
    reset,
    watch,
  } = useForm<ExtendedContactFormData>({
    resolver: zodResolver(contactFormSchema),
    defaultValues: {
      name: '',
      email: '',
      subject: '',
      projectType: '',
      message: '',
      acceptTerms: false,
      subscribeNewsletter: false,
    },
  })

  // Watch message field for character count
  const message = watch('message')
  useEffect(() => {
    setMessageLength(message?.length || 0)
  }, [message])

  const onSubmit = async (data: ContactFormData) => {
    setIsSubmitting(true)
    
    try {
      const response = await fetch('/api/contact', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(data),
      })

      const result = await response.json()

      if (response.ok) {
        showToast.success('Message sent successfully! We\'ll get back to you soon.')
        reset()
      } else {
        showToast.error(result.error || 'Failed to send message. Please try again.')
      }
    } catch (error) {
      showToast.error('An error occurred. Please try again later.')
      console.error('Contact form error:', error)
    } finally {
      setIsSubmitting(false)
    }
  }

  const contactInfo = [
    {
      icon: <Mail className="w-6 h-6" />,
      title: 'Email Us',
      info: 'duoteam.dev@email.com',
      subinfo: 'Available 24/7',
      color: 'from-blue-500 to-cyan-600',
      bgColor: 'bg-blue-50 dark:bg-blue-900/20',
      hoverColor: 'hover:border-blue-500'
    },
    {
      icon: <Phone className="w-6 h-6" />,
      title: 'Call Us',
      info: '+1 (555) 123-4567',
      subinfo: 'Mon-Fri, 9AM-6PM PST',
      color: 'from-green-500 to-emerald-600',
      bgColor: 'bg-green-50 dark:bg-green-900/20',
      hoverColor: 'hover:border-green-500'
    },
    {
      icon: <MapPin className="w-6 h-6" />,
      title: 'Location',
      info: 'San Francisco, CA',
      subinfo: 'Open to Remote Work',
      color: 'from-purple-500 to-pink-600',
      bgColor: 'bg-purple-50 dark:bg-purple-900/20',
      hoverColor: 'hover:border-purple-500'
    },
  ]

  const socialLinks = [
    { icon: <Github className="w-5 h-5" />, href: 'https://github.com', label: 'GitHub', color: 'hover:bg-gray-100 dark:hover:bg-gray-700' },
    { icon: <Linkedin className="w-5 h-5" />, href: 'https://linkedin.com', label: 'LinkedIn', color: 'hover:bg-blue-50 dark:hover:bg-blue-900/30' },
    { icon: <Twitter className="w-5 h-5" />, href: 'https://twitter.com', label: 'Twitter', color: 'hover:bg-sky-50 dark:hover:bg-sky-900/30' },
  ]

  const features = [
    {
      icon: <Clock className="w-5 h-5" />,
      title: '2-4 Hours',
      description: 'Fast Response Time',
      color: 'text-blue-600 dark:text-blue-400'
    },
    {
      icon: <Shield className="w-5 h-5" />,
      title: '100% Secure',
      description: 'Data Protection',
      color: 'text-green-600 dark:text-green-400'
    },
    {
      icon: <HeartHandshake className="w-5 h-5" />,
      title: 'Free Consultation',
      description: 'No Obligations',
      color: 'text-purple-600 dark:text-purple-400'
    },
    {
      icon: <Zap className="w-5 h-5" />,
      title: 'Quick Start',
      description: 'Begin in 48 Hours',
      color: 'text-orange-600 dark:text-orange-400'
    },
  ]

  const projectTypes = [
    { value: 'web-app', label: '🌐 Web Application', emoji: '🌐', popular: true },
    { value: 'mobile-app', label: '📱 Mobile Application', emoji: '📱', popular: true },
    { value: 'ai-ml', label: '🤖 AI/ML Solution', emoji: '🤖', popular: false },
    { value: 'blockchain', label: '⛓️ Blockchain Project', emoji: '⛓️', popular: false },
    { value: 'consulting', label: '💼 Consulting', emoji: '💼', popular: true },
    { value: 'other', label: '✨ Other', emoji: '✨', popular: false },
  ]

  return (
    <section id="contact" className="py-20 bg-gray-50/50 dark:bg-gray-900/50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5 }}
          className="text-center mb-16"
        >
          <h2 className="text-4xl md:text-5xl font-bold text-gradient mb-4">Get In Touch</h2>
          <p className="text-xl text-gray-600 dark:text-gray-400 max-w-3xl mx-auto">
            Have a project in mind? Whether it's a web app, mobile solution, or AI integration, 
            we're here to help bring your ideas to life
          </p>
        </motion.div>

        {/* Features Grid */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5 }}
          className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-16"
        >
          {features.map((feature, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, scale: 0.9 }}
              whileInView={{ opacity: 1, scale: 1 }}
              viewport={{ once: true }}
              transition={{ duration: 0.3, delay: index * 0.1 }}
              className="bg-white dark:bg-gray-800/50 backdrop-blur-sm border border-gray-200 dark:border-gray-700 rounded-2xl p-5 text-center hover:border-gray-300 dark:hover:border-gray-600 transition-all duration-300 hover:shadow-lg"
            >
              <div className={`w-12 h-12 ${feature.color} mx-auto mb-3 flex items-center justify-center`}>
                {feature.icon}
              </div>
              <h4 className="text-sm font-bold text-gray-900 dark:text-white mb-1">
                {feature.title}
              </h4>
              <p className="text-xs text-gray-600 dark:text-gray-400">
                {feature.description}
              </p>
            </motion.div>
          ))}
        </motion.div>

        <div className="grid lg:grid-cols-5 gap-12">
          {/* Contact Info - Left Side (2 columns) */}
          <motion.div
            initial={{ opacity: 0, x: -20 }}
            whileInView={{ opacity: 1, x: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5 }}
            className="lg:col-span-2 space-y-8"
          >
            {/* Title */}
            <div>
              <h3 className="text-3xl font-bold text-gray-900 dark:text-white mb-2">Let's Talk</h3>
              <p className="text-gray-600 dark:text-gray-400">
                Choose your preferred way to connect with us
              </p>
            </div>

            {/* Contact Cards */}
            <div className="space-y-4">
              {contactInfo.map((item, index) => (
                <motion.div
                  key={index}
                  initial={{ opacity: 0, x: -20 }}
                  whileInView={{ opacity: 1, x: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.5, delay: 0.1 * index }}
                  whileHover={{ x: 5, scale: 1.02 }}
                  className={`group relative overflow-hidden p-5 bg-white dark:bg-gray-800/50 backdrop-blur-sm border-2 border-gray-200 dark:border-gray-700 rounded-2xl ${item.hoverColor} transition-all duration-300 cursor-pointer`}
                >
                  {/* Background Gradient on Hover */}
                  <div className={`absolute inset-0 bg-gradient-to-r ${item.color} opacity-0 group-hover:opacity-5 transition-opacity duration-300`} />
                  
                  <div className="relative flex items-start gap-4">
                    <div className={`flex-shrink-0 w-14 h-14 ${item.bgColor} rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300`}>
                      <div className={`bg-gradient-to-br ${item.color} bg-clip-text text-transparent`}>
                        {item.icon}
                      </div>
                    </div>
                    <div className="flex-1 min-w-0">
                      <h4 className="text-gray-900 dark:text-white font-bold mb-1 text-lg">
                        {item.title}
                      </h4>
                      <p className="text-gray-900 dark:text-gray-100 font-medium mb-1 break-all">
                        {item.info}
                      </p>
                      <p className="text-gray-600 dark:text-gray-400 text-sm flex items-center gap-1.5">
                        <CheckCircle2 className="w-3.5 h-3.5 text-green-500" />
                        {item.subinfo}
                      </p>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>

            {/* Social Links */}
            <div className="pt-4">
              <h4 className="text-gray-900 dark:text-white font-bold mb-4 text-lg flex items-center gap-2">
                <Globe className="w-5 h-5" />
                Follow Our Journey
              </h4>
              <div className="flex gap-3">
                {socialLinks.map((social, index) => (
                  <motion.a
                    key={index}
                    href={social.href}
                    target="_blank"
                    rel="noopener noreferrer"
                    whileHover={{ scale: 1.1, rotate: 5 }}
                    whileTap={{ scale: 0.95 }}
                    className={`w-14 h-14 flex items-center justify-center bg-white dark:bg-gray-800/50 backdrop-blur-sm border-2 border-gray-200 dark:border-gray-700 rounded-xl text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:border-gray-400 dark:hover:border-gray-500 ${social.color} hover:shadow-lg transition-all duration-300`}
                    aria-label={social.label}
                  >
                    {social.icon}
                  </motion.a>
                ))}
              </div>
            </div>

            {/* Trust Badge */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5, delay: 0.3 }}
              className="relative p-6 bg-gradient-to-br from-red-500/10 via-purple-500/10 to-blue-600/10 border-2 border-gray-200 dark:border-gray-700 rounded-2xl overflow-hidden"
            >
              <div className="absolute top-0 right-0 w-32 h-32 bg-gradient-to-br from-red-500/20 to-blue-600/20 rounded-full blur-2xl" />
              <div className="relative">
                <div className="flex items-center gap-2 mb-3">
                  <Shield className="w-5 h-5 text-green-500" />
                  <span className="font-bold text-gray-900 dark:text-white">Trusted by Clients</span>
                </div>
                <p className="text-gray-700 dark:text-gray-300 text-sm leading-relaxed">
                  <strong className="text-gradient">Fast Response:</strong> 2-4 hours during business hours. 
                  For urgent matters, call us directly. Your data is 100% secure and confidential.
                </p>
                <div className="flex items-center gap-2 mt-4 pt-4 border-t border-gray-300 dark:border-gray-600">
                  <div className="flex -space-x-2">
                    {[1, 2, 3, 4].map((i) => (
                      <div key={i} className="w-8 h-8 rounded-full bg-gradient-to-br from-red-500 to-blue-600 border-2 border-white dark:border-gray-800 flex items-center justify-center text-white text-xs font-bold">
                        {i}
                      </div>
                    ))}
                  </div>
                  <div className="text-sm">
                    <p className="font-semibold text-gray-900 dark:text-white">50+ Happy Clients</p>
                    <div className="flex items-center gap-1">
                      {[1, 2, 3, 4, 5].map((i) => (
                        <Star key={i} className="w-3 h-3 fill-yellow-400 text-yellow-400" />
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            </motion.div>
          </motion.div>

          {/* Contact Form - Right Side (3 columns) */}
          <motion.div
            initial={{ opacity: 0, x: 20 }}
            whileInView={{ opacity: 1, x: 0 }}
            viewport={{ once: true }}
            transition={{ duration: 0.5 }}
            className="lg:col-span-3"
          >
            {/* Form Container */}
            <div className="sticky top-8 p-8 lg:p-10 bg-white dark:bg-gray-800/50 backdrop-blur-sm border-2 border-gray-200 dark:border-gray-700 rounded-3xl shadow-2xl">
              {/* Form Header */}
              <div className="mb-8">
                <h3 className="text-3xl font-bold text-gray-900 dark:text-white mb-2 flex items-center gap-2">
                  <Rocket className="w-7 h-7 text-gradient" />
                  Start Your Project
                </h3>
                <p className="text-gray-600 dark:text-gray-400">
                  Fill in the details and we'll get back to you within 2-4 hours
                </p>
              </div>

              <form onSubmit={handleSubmit(onSubmit)} className="space-y-6">
                {/* Name & Email Row */}
                <div className="grid md:grid-cols-2 gap-6">
                  {/* Name Field - NEW DESIGN */}
                  <motion.div
                    initial={{ opacity: 0, y: 10 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.1 }}
                    className="group"
                  >
                    <label htmlFor="name" className="block text-sm font-bold text-gray-900 dark:text-white mb-2.5">
                      <span className="flex items-center gap-2">
                        <div className="w-6 h-6 rounded-lg bg-gradient-to-br from-red-500 to-orange-500 flex items-center justify-center">
                          <User className="w-3.5 h-3.5 text-white" />
                        </div>
                        Full Name <span className="text-red-500">*</span>
                      </span>
                    </label>
                    <div className="relative">
                      <Controller
                        name="name"
                        control={control}
                        render={({ field }) => (
                          <input
                            {...field}
                            id="name"
                            type="text"
                            placeholder="Enter your full name"
                            className={`w-full px-5 py-4 bg-gray-50 dark:bg-gray-900/50 border-2 rounded-xl
                              text-gray-900 dark:text-white placeholder:text-gray-400 dark:placeholder:text-gray-600
                              transition-all duration-300 outline-none
                              ${errors.name 
                                ? 'border-red-500 focus:border-red-600 focus:ring-4 focus:ring-red-500/20' 
                                : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600 focus:border-red-500 focus:ring-4 focus:ring-red-500/10'
                              }
                              group-hover:shadow-lg`}
                          />
                        )}
                      />
                      {!errors.name && (
                        <div className="absolute right-4 top-1/2 -translate-y-1/2 opacity-0 group-hover:opacity-100 transition-opacity">
                          <div className="w-2 h-2 rounded-full bg-gradient-to-r from-red-500 to-orange-500 animate-pulse"></div>
                        </div>
                      )}
                    </div>
                    <AnimatePresence>
                      {errors.name && (
                        <motion.p 
                          initial={{ opacity: 0, height: 0 }}
                          animate={{ opacity: 1, height: 'auto' }}
                          exit={{ opacity: 0, height: 0 }}
                          className="mt-2.5 text-sm text-red-600 dark:text-red-400 flex items-center gap-2 font-medium"
                        >
                          <span className="inline-block w-1.5 h-1.5 rounded-full bg-red-600 animate-pulse"></span>
                          {errors.name.message}
                        </motion.p>
                      )}
                    </AnimatePresence>
                  </motion.div>

                  {/* Email Field - NEW DESIGN */}
                  <motion.div
                    initial={{ opacity: 0, y: 10 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ delay: 0.2 }}
                    className="group"
                  >
                    <label htmlFor="email" className="block text-sm font-bold text-gray-900 dark:text-white mb-2.5">
                      <span className="flex items-center gap-2">
                        <div className="w-6 h-6 rounded-lg bg-gradient-to-br from-blue-500 to-cyan-500 flex items-center justify-center">
                          <Mail className="w-3.5 h-3.5 text-white" />
                        </div>
                        Email Address <span className="text-red-500">*</span>
                      </span>
                    </label>
                    <div className="relative">
                      <Controller
                        name="email"
                        control={control}
                        render={({ field }) => (
                          <input
                            {...field}
                            id="email"
                            type="email"
                            placeholder="your@email.com"
                            className={`w-full px-5 py-4 bg-gray-50 dark:bg-gray-900/50 border-2 rounded-xl
                              text-gray-900 dark:text-white placeholder:text-gray-400 dark:placeholder:text-gray-600
                              transition-all duration-300 outline-none
                              ${errors.email 
                                ? 'border-red-500 focus:border-red-600 focus:ring-4 focus:ring-red-500/20' 
                                : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600 focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10'
                              }
                              group-hover:shadow-lg`}
                          />
                        )}
                      />
                      {!errors.email && (
                        <div className="absolute right-4 top-1/2 -translate-y-1/2 opacity-0 group-hover:opacity-100 transition-opacity">
                          <div className="w-2 h-2 rounded-full bg-gradient-to-r from-blue-500 to-cyan-500 animate-pulse"></div>
                        </div>
                      )}
                    </div>
                    <AnimatePresence>
                      {errors.email && (
                        <motion.p 
                          initial={{ opacity: 0, height: 0 }}
                          animate={{ opacity: 1, height: 'auto' }}
                          exit={{ opacity: 0, height: 0 }}
                          className="mt-2.5 text-sm text-red-600 dark:text-red-400 flex items-center gap-2 font-medium"
                        >
                          <span className="inline-block w-1.5 h-1.5 rounded-full bg-red-600 animate-pulse"></span>
                          {errors.email.message}
                        </motion.p>
                      )}
                    </AnimatePresence>
                  </motion.div>
                </div>

                {/* Subject Field - FULL WIDTH NEW DESIGN */}
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.3 }}
                  className="group"
                >
                  <label htmlFor="subject" className="block text-sm font-bold text-gray-900 dark:text-white mb-2.5">
                    <span className="flex items-center gap-2">
                      <div className="w-6 h-6 rounded-lg bg-gradient-to-br from-purple-500 to-pink-500 flex items-center justify-center">
                        <FileText className="w-3.5 h-3.5 text-white" />
                      </div>
                      Project Subject <span className="text-red-500">*</span>
                    </span>
                  </label>
                  <div className="relative">
                    <Controller
                      name="subject"
                      control={control}
                      render={({ field }) => (
                        <input
                          {...field}
                          id="subject"
                          type="text"
                          placeholder="E.g., E-commerce Platform Development"
                          className={`w-full px-5 py-4 bg-gray-50 dark:bg-gray-900/50 border-2 rounded-xl
                            text-gray-900 dark:text-white placeholder:text-gray-400 dark:placeholder:text-gray-600
                            transition-all duration-300 outline-none
                            ${errors.subject 
                              ? 'border-red-500 focus:border-red-600 focus:ring-4 focus:ring-red-500/20' 
                              : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600 focus:border-purple-500 focus:ring-4 focus:ring-purple-500/10'
                            }
                            group-hover:shadow-lg`}
                        />
                      )}
                    />
                    {!errors.subject && (
                      <div className="absolute right-4 top-1/2 -translate-y-1/2 opacity-0 group-hover:opacity-100 transition-opacity">
                        <div className="w-2 h-2 rounded-full bg-gradient-to-r from-purple-500 to-pink-500 animate-pulse"></div>
                      </div>
                    )}
                  </div>
                  <AnimatePresence>
                    {errors.subject && (
                      <motion.p 
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: 'auto' }}
                        exit={{ opacity: 0, height: 0 }}
                        className="mt-2.5 text-sm text-red-600 dark:text-red-400 flex items-center gap-2 font-medium"
                      >
                        <span className="inline-block w-1.5 h-1.5 rounded-full bg-red-600 animate-pulse"></span>
                        {errors.subject.message}
                      </motion.p>
                    )}
                  </AnimatePresence>
                </motion.div>

                {/* Project Type Field - NEW SELECT DESIGN */}
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.4 }}
                  className="group"
                >
                  <label htmlFor="projectType" className="block text-sm font-bold text-gray-900 dark:text-white mb-2.5">
                    <span className="flex items-center gap-2">
                      <div className="w-6 h-6 rounded-lg bg-gradient-to-br from-green-500 to-emerald-500 flex items-center justify-center">
                        <Sparkles className="w-3.5 h-3.5 text-white" />
                      </div>
                      Project Type
                    </span>
                  </label>
                  <div className="relative">
                    <Controller
                      name="projectType"
                      control={control}
                      render={({ field }) => (
                        <select
                          {...field}
                          id="projectType"
                          className={`w-full px-5 py-4 bg-gray-50 dark:bg-gray-900/50 border-2 rounded-xl
                            text-gray-900 dark:text-white appearance-none cursor-pointer
                            transition-all duration-300 outline-none
                            ${errors.projectType 
                              ? 'border-red-500 focus:border-red-600 focus:ring-4 focus:ring-red-500/20' 
                              : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600 focus:border-green-500 focus:ring-4 focus:ring-green-500/10'
                            }
                            group-hover:shadow-lg`}
                        >
                          <option value="" disabled className="text-gray-400">
                            Select project type...
                          </option>
                          {projectTypes.map((type) => (
                            <option key={type.value} value={type.value} className="py-2">
                              {type.label} {type.popular ? '⭐' : ''}
                            </option>
                          ))}
                        </select>
                      )}
                    />
                    <div className="absolute right-4 top-1/2 -translate-y-1/2 pointer-events-none">
                      <ArrowRight className="w-4 h-4 text-gray-400 rotate-90" />
                    </div>
                  </div>
                  <AnimatePresence>
                    {errors.projectType && (
                      <motion.p 
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: 'auto' }}
                        exit={{ opacity: 0, height: 0 }}
                        className="mt-2.5 text-sm text-red-600 dark:text-red-400 flex items-center gap-2 font-medium"
                      >
                        <span className="inline-block w-1.5 h-1.5 rounded-full bg-red-600 animate-pulse"></span>
                        {errors.projectType.message}
                      </motion.p>
                    )}
                  </AnimatePresence>
                </motion.div>

                {/* Message Field - NEW TEXTAREA DESIGN */}
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.5 }}
                  className="group"
                >
                  <div className="flex items-center justify-between mb-2.5">
                    <label htmlFor="message" className="flex items-center gap-2 text-gray-900 dark:text-white font-bold text-sm">
                      <div className="w-6 h-6 rounded-lg bg-gradient-to-br from-orange-500 to-yellow-500 flex items-center justify-center">
                        <MessageSquare className="w-3.5 h-3.5 text-white" />
                      </div>
                      Your Message <span className="text-red-500">*</span>
                    </label>
                    <span className={`text-xs font-bold px-3 py-1.5 rounded-full transition-all ${
                      messageLength > maxMessageLength 
                        ? 'bg-red-100 text-red-700 dark:bg-red-900/30 dark:text-red-400' 
                        : messageLength > maxMessageLength * 0.8 
                        ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-400' 
                        : 'bg-gray-100 text-gray-600 dark:bg-gray-800 dark:text-gray-400'
                    }`}>
                      {messageLength} / {maxMessageLength}
                    </span>
                  </div>
                  <Controller
                    name="message"
                    control={control}
                    render={({ field }) => (
                      <textarea
                        {...field}
                        id="message"
                        rows={7}
                        maxLength={maxMessageLength}
                        placeholder="Tell us about your project:&#10;&#10;• What are your goals?&#10;• Timeline and budget?&#10;• Any specific requirements?&#10;• What challenges are you facing?"
                        className={`w-full px-5 py-4 bg-gray-50 dark:bg-gray-900/50 border-2 rounded-xl
                          text-gray-900 dark:text-white placeholder:text-gray-400 dark:placeholder:text-gray-600
                          transition-all duration-300 outline-none resize-none
                          ${errors.message 
                            ? 'border-red-500 focus:border-red-600 focus:ring-4 focus:ring-red-500/20' 
                            : 'border-gray-200 dark:border-gray-700 hover:border-gray-300 dark:hover:border-gray-600 focus:border-orange-500 focus:ring-4 focus:ring-orange-500/10'
                          }
                          group-hover:shadow-lg`}
                      />
                    )}
                  />
                  <AnimatePresence>
                    {errors.message && (
                      <motion.p 
                        initial={{ opacity: 0, height: 0 }}
                        animate={{ opacity: 1, height: 'auto' }}
                        exit={{ opacity: 0, height: 0 }}
                        className="mt-2.5 text-sm text-red-600 dark:text-red-400 flex items-center gap-2 font-medium"
                      >
                        <span className="inline-block w-1.5 h-1.5 rounded-full bg-red-600 animate-pulse"></span>
                        {errors.message.message}
                      </motion.p>
                    )}
                  </AnimatePresence>
                </motion.div>

                {/* Checkboxes - NEW DESIGN */}
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.6 }}
                  className="space-y-4 p-6 bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-900/50 dark:to-gray-900/30 rounded-2xl border-2 border-gray-200 dark:border-gray-700"
                >
                  <Controller
                    name="acceptTerms"
                    control={control}
                    render={({ field: { value, ...field } }) => (
                      <div className="flex items-start gap-3 group cursor-pointer">
                        <input
                          type="checkbox"
                          {...field}
                          checked={!!value}
                          id="acceptTerms"
                          className="mt-1 w-5 h-5 rounded-md border-2 border-gray-300 dark:border-gray-600 
                            text-red-500 focus:ring-4 focus:ring-red-500/20 transition-all cursor-pointer
                            checked:bg-gradient-to-br checked:from-red-500 checked:to-orange-500 checked:border-transparent"
                        />
                        <label htmlFor="acceptTerms" className="text-sm text-gray-700 dark:text-gray-300 leading-relaxed cursor-pointer select-none">
                          I accept the{' '}
                          <a href="#" className="text-red-500 hover:text-red-600 dark:text-red-400 dark:hover:text-red-300 font-bold underline underline-offset-2 transition-colors">
                            Terms & Conditions
                          </a>
                          {' '}and{' '}
                          <a href="#" className="text-blue-500 hover:text-blue-600 dark:text-blue-400 dark:hover:text-blue-300 font-bold underline underline-offset-2 transition-colors">
                            Privacy Policy
                          </a>
                        </label>
                      </div>
                    )}
                  />

                  <Controller
                    name="subscribeNewsletter"
                    control={control}
                    render={({ field: { value, ...field } }) => (
                      <div className="flex items-start gap-3 group cursor-pointer">
                        <input
                          type="checkbox"
                          {...field}
                          checked={!!value}
                          id="subscribeNewsletter"
                          className="mt-1 w-5 h-5 rounded-md border-2 border-gray-300 dark:border-gray-600 
                            text-blue-500 focus:ring-4 focus:ring-blue-500/20 transition-all cursor-pointer
                            checked:bg-gradient-to-br checked:from-blue-500 checked:to-cyan-500 checked:border-transparent"
                        />
                        <label htmlFor="subscribeNewsletter" className="text-sm text-gray-700 dark:text-gray-300 leading-relaxed cursor-pointer select-none">
                          📬 Subscribe to our newsletter for project updates, tech insights, and exclusive offers
                        </label>
                      </div>
                    )}
                  />
                </motion.div>

                {/* Submit Button - NEW DESIGN */}
                <motion.div
                  initial={{ opacity: 0, y: 10 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: 0.7 }}
                  className="pt-2"
                >
                  <button
                    type="submit"
                    disabled={isSubmitting}
                    className="group relative w-full px-8 py-5 bg-gradient-to-r from-red-500 via-purple-500 to-blue-600 text-white rounded-2xl font-bold text-lg
                      hover:shadow-2xl hover:shadow-purple-500/30 hover:scale-[1.02] active:scale-[0.98]
                      disabled:opacity-50 disabled:cursor-not-allowed disabled:hover:scale-100
                      transition-all duration-300 overflow-hidden"
                  >
                    {/* Shimmer Effect */}
                    <div className="absolute inset-0 bg-gradient-to-r from-transparent via-white/20 to-transparent 
                      translate-x-[-200%] group-hover:translate-x-[200%] transition-transform duration-1000"></div>
                    
                    <span className="relative flex items-center justify-center gap-3">
                      {isSubmitting ? (
                        <>
                          <div className="w-5 h-5 border-3 border-white/30 border-t-white rounded-full animate-spin"></div>
                          Sending your message...
                        </>
                      ) : (
                        <>
                          <Rocket className="w-6 h-6 group-hover:translate-x-1 transition-transform" />
                          Send Message
                          <Send className="w-5 h-5 group-hover:translate-x-1 group-hover:-translate-y-1 transition-transform" />
                        </>
                      )}
                    </span>
                  </button>
                  
                  {/* Trust indicators */}
                  <div className="mt-4 flex items-center justify-center gap-6 text-xs text-gray-600 dark:text-gray-400">
                    <div className="flex items-center gap-1.5">
                      <Shield className="w-4 h-4 text-green-500" />
                      <span>100% Secure</span>
                    </div>
                    <div className="flex items-center gap-1.5">
                      <Clock className="w-4 h-4 text-blue-500" />
                      <span>2-4h Response</span>
                    </div>
                    <div className="flex items-center gap-1.5">
                      <CheckCircle2 className="w-4 h-4 text-purple-500" />
                      <span>No Spam</span>
                    </div>
                  </div>
                </motion.div>
              </form>
            </div>
          </motion.div>
        </div>

        {/* CTA Section */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5, delay: 0.3 }}
          className="mt-20 text-center bg-gradient-to-r from-red-500/10 to-blue-600/10 border border-gray-200 dark:border-gray-700 rounded-2xl p-12"
        >
          <h3 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">Ready to Start Your Project?</h3>
          <p className="text-gray-600 dark:text-gray-400 mb-8 max-w-2xl mx-auto">
            From concept to deployment, we'll guide you through every step of the development process. 
            Let's turn your vision into reality with cutting-edge technology and best practices.
          </p>
          <div className="flex flex-col sm:flex-row items-center justify-center gap-4">
            <a
              href="#contact"
              className="px-8 py-4 bg-gradient-to-r from-red-500 to-blue-600 text-white rounded-full font-semibold hover:shadow-glow-duo transition-all duration-300 transform hover:scale-105"
            >
              Get Started Today
            </a>
            <a
              href="#about"
              className="px-8 py-4 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-white rounded-full font-semibold hover:border-gray-400 dark:hover:border-gray-600 hover:bg-gray-100/50 dark:hover:bg-gray-800/50 transition-all duration-300"
            >
              Learn More About Us
            </a>
          </div>
        </motion.div>
      </div>
    </section>
  )
}
