'use client'

import { Phone, X } from 'lucide-react'
import { motion, AnimatePresence } from 'framer-motion'
import { useState, useEffect } from 'react'
import Link from 'next/link'
import { useTranslations } from 'next-intl'

export function FloatingCTA() {
  const t = useTranslations('floatingCTA')
  const [isVisible, setIsVisible] = useState(false)
  const [isDismissed, setIsDismissed] = useState(false)

  useEffect(() => {
    // Show floating CTA after 3 seconds of scrolling
    const handleScroll = () => {
      if (window.scrollY > 500 && !isDismissed) {
        setIsVisible(true)
      }
    }

    // Show immediately on mobile for better UX
    if (window.innerWidth < 768) {
      setTimeout(() => setIsVisible(true), 2000)
    }

    window.addEventListener('scroll', handleScroll)
    return () => window.removeEventListener('scroll', handleScroll)
  }, [isDismissed])

  const handleDismiss = () => {
    setIsVisible(false)
    setIsDismissed(true)
    // Store dismissal in localStorage to remember user preference
    localStorage.setItem('floatingCTA-dismissed', 'true')
  }

  // Check if user dismissed it before
  useEffect(() => {
    const dismissed = localStorage.getItem('floatingCTA-dismissed')
    if (dismissed === 'true') {
      setIsDismissed(true)
    }
  }, [])

  return (
    <AnimatePresence>
      {isVisible && !isDismissed && (
        <motion.div
          initial={{ opacity: 0, scale: 0.8, y: 20 }}
          animate={{ opacity: 1, scale: 1, y: 0 }}
          exit={{ opacity: 0, scale: 0.8, y: 20 }}
          transition={{ duration: 0.3 }}
          className="fixed bottom-6 right-6 z-50 flex flex-col gap-3"
        >
          {/* Close Button */}
          <button
            onClick={handleDismiss}
            className="self-end w-8 h-8 flex items-center justify-center bg-gray-800/90 backdrop-blur-sm rounded-full text-white hover:bg-gray-700 transition-colors shadow-lg"
            aria-label={t('close')}
          >
            <X className="w-4 h-4" />
          </button>

          {/* CTA Button */}
          <Link
            href="#contact"
            className="group flex items-center gap-3 px-6 py-4 bg-gradient-to-r from-red-500 to-blue-600 text-white rounded-full font-semibold shadow-glow-duo hover:shadow-glow-duo-lg transition-all duration-300 transform hover:scale-105"
          >
            <Phone className="w-5 h-5 group-hover:animate-pulse" />
            <span className="hidden sm:inline">{t('bookConsultation')}</span>
            <span className="sm:hidden">{t('letsTalk')}</span>
          </Link>

          {/* Urgency Badge */}
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ delay: 0.3 }}
            className="px-3 py-1 bg-white dark:bg-gray-800 rounded-full text-xs text-gray-700 dark:text-gray-300 shadow-lg text-center"
          >
            ⚡ {t('responseTime')}
          </motion.div>
        </motion.div>
      )}
    </AnimatePresence>
  )
}

