'use client'

import { useState, useTransition } from 'react'
import { useRouter, usePathname } from 'next/navigation'
import { useLocale } from 'next-intl'
import { Globe, Check } from 'lucide-react'
import { locales, localeNames, localeFlags, type Locale } from '@/i18n'
import { AnimatePresence, motion } from 'framer-motion'

export function LanguageSwitcher() {
  const [isOpen, setIsOpen] = useState(false)
  const [isPending, startTransition] = useTransition()
  const router = useRouter()
  const pathname = usePathname()
  const locale = useLocale() as Locale

  const handleLocaleChange = (newLocale: Locale) => {
    setIsOpen(false)
    
    startTransition(() => {
      // Replace the current locale in the pathname
      const pathWithoutLocale = pathname.replace(/^\/(en|ru|ua|ka)/, '')
      router.replace(`/${newLocale}${pathWithoutLocale}`)
    })
  }

  return (
    <div className="relative">
      {/* Trigger Button */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="flex items-center gap-2 px-3 py-2 rounded-lg border border-gray-700 dark:border-gray-700 border-gray-300 bg-gray-800/50 dark:bg-gray-800/50 bg-white/80 backdrop-blur-sm text-gray-300 dark:text-gray-300 text-gray-700 hover:border-gray-600 dark:hover:border-gray-600 hover:border-gray-400 transition-all duration-300 hover:shadow-glow-duo"
        disabled={isPending}
      >
        <Globe className="w-4 h-4" />
        <span className="hidden sm:inline text-sm font-medium">
          {localeFlags[locale]} {localeNames[locale]}
        </span>
        <span className="sm:hidden text-sm font-medium">
          {localeFlags[locale]}
        </span>
      </button>

      {/* Dropdown Menu */}
      <AnimatePresence>
        {isOpen && (
          <>
            {/* Backdrop */}
            <div
              className="fixed inset-0 z-40"
              onClick={() => setIsOpen(false)}
            />

            {/* Menu */}
            <motion.div
              initial={{ opacity: 0, scale: 0.95, y: -10 }}
              animate={{ opacity: 1, scale: 1, y: 0 }}
              exit={{ opacity: 0, scale: 0.95, y: -10 }}
              transition={{ duration: 0.2 }}
              className="absolute right-0 mt-2 w-48 bg-gray-800 dark:bg-gray-800 bg-white border border-gray-700 dark:border-gray-700 border-gray-300 rounded-lg shadow-2xl overflow-hidden z-50 backdrop-blur-md"
            >
              <div className="py-1">
                {locales.map((loc) => (
                  <button
                    key={loc}
                    onClick={() => handleLocaleChange(loc)}
                    disabled={isPending}
                    className={`w-full flex items-center justify-between px-4 py-3 text-left transition-all duration-200 ${
                      locale === loc
                        ? 'bg-gradient-to-r from-red-500/20 to-blue-600/20 text-white dark:text-white text-gray-900'
                        : 'text-gray-300 dark:text-gray-300 text-gray-700 hover:bg-gray-700/50 dark:hover:bg-gray-700/50 hover:bg-gray-100'
                    }`}
                  >
                    <div className="flex items-center gap-3">
                      <span className="text-xl">{localeFlags[loc]}</span>
                      <span className="font-medium">{localeNames[loc]}</span>
                    </div>
                    {locale === loc && (
                      <Check className="w-4 h-4 text-gradient" />
                    )}
                  </button>
                ))}
              </div>

              {/* Loading Indicator */}
              {isPending && (
                <div className="absolute inset-0 bg-gray-900/50 backdrop-blur-sm flex items-center justify-center">
                  <div className="w-6 h-6 border-2 border-gray-300 border-t-transparent rounded-full animate-spin"></div>
                </div>
              )}
            </motion.div>
          </>
        )}
      </AnimatePresence>
    </div>
  )
}

