'use client'

import { Modal } from 'antd'
import { motion } from 'framer-motion'
import { X, Github, ExternalLink, Calendar, Tag, Users } from 'lucide-react'
import Image from 'next/image'

interface Project {
  title: string
  description: string
  longDescription?: string
  image: string
  tags: string[]
  category: string
  github: string | null
  demo: string | null
  features?: string[]
  technologies?: string[]
  teamSize?: number
  duration?: string
  completedDate?: string
}

interface ProjectModalProps {
  project: Project | null
  isOpen: boolean
  onClose: () => void
}

export function ProjectModal({ project, isOpen, onClose }: ProjectModalProps) {
  if (!project) return null

  return (
    <Modal
      open={isOpen}
      onCancel={onClose}
      footer={null}
      width={900}
      centered
      className="project-modal"
      closeIcon={<X className="w-5 h-5" />}
    >
      <div className="py-4">
        {/* Project Image */}
        <div className="relative h-64 md:h-96 rounded-xl overflow-hidden mb-6">
          <img
            src={project.image}
            alt={project.title}
            className="w-full h-full object-cover"
          />
          <div className="absolute top-4 left-4">
            <span className="px-3 py-1 bg-gradient-to-r from-red-500 to-blue-600 text-white text-sm font-semibold rounded-full shadow-glow-duo">
              {project.category}
            </span>
          </div>
        </div>

        {/* Project Title & Description */}
        <div className="mb-6">
          <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-3">{project.title}</h2>
          <p className="text-gray-700 dark:text-gray-300 text-lg mb-4">
            {project.longDescription || project.description}
          </p>
        </div>

        {/* Project Meta Info */}
        {(project.teamSize || project.duration || project.completedDate) && (
          <div className="grid grid-cols-1 sm:grid-cols-3 gap-4 mb-6">
            {project.teamSize && (
              <div className="flex items-center gap-2 text-gray-600 dark:text-gray-400">
                <Users className="w-5 h-5" />
                <span>{project.teamSize} Team Members</span>
              </div>
            )}
            {project.duration && (
              <div className="flex items-center gap-2 text-gray-600 dark:text-gray-400">
                <Calendar className="w-5 h-5" />
                <span>{project.duration}</span>
              </div>
            )}
            {project.completedDate && (
              <div className="flex items-center gap-2 text-gray-600 dark:text-gray-400">
                <Calendar className="w-5 h-5" />
                <span>Completed {project.completedDate}</span>
              </div>
            )}
          </div>
        )}

        {/* Features */}
        {project.features && project.features.length > 0 && (
          <div className="mb-6">
            <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-3">Key Features</h3>
            <ul className="grid grid-cols-1 sm:grid-cols-2 gap-2">
              {project.features.map((feature, index) => (
                <li key={index} className="flex items-start gap-2 text-gray-700 dark:text-gray-300">
                  <span className="text-gradient font-bold">•</span>
                  <span>{feature}</span>
                </li>
              ))}
            </ul>
          </div>
        )}

        {/* Technologies */}
        <div className="mb-6">
          <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-3 flex items-center gap-2">
            <Tag className="w-5 h-5" />
            Technologies Used
          </h3>
          <div className="flex flex-wrap gap-2">
            {(project.technologies || project.tags).map((tech, index) => (
              <span
                key={index}
                className="px-3 py-1 bg-gray-100 dark:bg-gray-700/50 text-gray-700 dark:text-gray-300 text-sm rounded-md hover:bg-gradient-to-r hover:from-red-500/20 hover:to-blue-600/20 transition-colors"
              >
                {tech}
              </span>
            ))}
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex flex-col sm:flex-row gap-3">
          {project.github && (
            <a
              href={project.github}
              target="_blank"
              rel="noopener noreferrer"
              className="flex-1 px-6 py-3 bg-gray-100 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 text-gray-900 dark:text-white rounded-lg font-semibold hover:border-gray-300 dark:hover:border-gray-600 hover:bg-gray-200 dark:hover:bg-gray-700 transition-all duration-300 flex items-center justify-center gap-2"
            >
              <Github className="w-5 h-5" />
              View on GitHub
            </a>
          )}
          {project.demo && (
            <a
              href={project.demo}
              target="_blank"
              rel="noopener noreferrer"
              className="flex-1 px-6 py-3 bg-gradient-to-r from-red-500 to-blue-600 text-white rounded-lg font-semibold hover:shadow-glow-duo transition-all duration-300 flex items-center justify-center gap-2"
            >
              <ExternalLink className="w-5 h-5" />
              Live Demo
            </a>
          )}
        </div>
      </div>
    </Modal>
  )
}

