'use client'

import React from 'react'
import { motion } from 'framer-motion'

interface SkeletonLoaderProps {
  type?: 'card' | 'text' | 'avatar' | 'stat' | 'project' | 'skill'
  count?: number
  className?: string
}

export function SkeletonLoader({ type = 'card', count = 1, className = '' }: SkeletonLoaderProps) {
  const skeletons = Array.from({ length: count }, (_, i) => i)

  const renderSkeleton = () => {
    switch (type) {
      case 'card':
        return (
          <div className={`bg-gray-800/50 border border-gray-700 rounded-2xl p-6 ${className}`}>
            <div className="skeleton-loader h-6 w-3/4 rounded-lg mb-4" />
            <div className="skeleton-loader h-4 w-full rounded-lg mb-2" />
            <div className="skeleton-loader h-4 w-5/6 rounded-lg mb-4" />
            <div className="flex gap-2 mt-4">
              <div className="skeleton-loader h-6 w-16 rounded-full" />
              <div className="skeleton-loader h-6 w-20 rounded-full" />
              <div className="skeleton-loader h-6 w-16 rounded-full" />
            </div>
          </div>
        )

      case 'project':
        return (
          <div className={`bg-gray-800/50 border border-gray-700 rounded-2xl overflow-hidden ${className}`}>
            <div className="skeleton-loader h-48 w-full" />
            <div className="p-6">
              <div className="skeleton-loader h-6 w-3/4 rounded-lg mb-3" />
              <div className="skeleton-loader h-4 w-full rounded-lg mb-2" />
              <div className="skeleton-loader h-4 w-4/5 rounded-lg mb-4" />
              <div className="flex gap-2">
                <div className="skeleton-loader h-6 w-16 rounded-full" />
                <div className="skeleton-loader h-6 w-20 rounded-full" />
                <div className="skeleton-loader h-6 w-18 rounded-full" />
              </div>
            </div>
          </div>
        )

      case 'stat':
        return (
          <div className={`bg-gray-800/50 border border-gray-700 rounded-2xl p-6 text-center ${className}`}>
            <div className="skeleton-loader h-12 w-24 mx-auto rounded-lg mb-3" />
            <div className="skeleton-loader h-4 w-32 mx-auto rounded-lg mb-2" />
            <div className="skeleton-loader h-3 w-24 mx-auto rounded-lg" />
          </div>
        )

      case 'skill':
        return (
          <div className={`bg-gray-800/50 border border-gray-700 rounded-2xl p-6 ${className}`}>
            <div className="skeleton-loader h-6 w-2/3 rounded-lg mb-2" />
            <div className="skeleton-loader h-4 w-full rounded-lg mb-6" />
            <div className="space-y-4">
              {[...Array(4)].map((_, i) => (
                <div key={i}>
                  <div className="flex justify-between mb-2">
                    <div className="skeleton-loader h-4 w-20 rounded-lg" />
                    <div className="skeleton-loader h-4 w-12 rounded-lg" />
                  </div>
                  <div className="skeleton-loader h-2 w-full rounded-full" />
                </div>
              ))}
            </div>
          </div>
        )

      case 'text':
        return (
          <div className={`space-y-2 ${className}`}>
            <div className="skeleton-loader h-4 w-full rounded-lg" />
            <div className="skeleton-loader h-4 w-5/6 rounded-lg" />
            <div className="skeleton-loader h-4 w-4/5 rounded-lg" />
          </div>
        )

      case 'avatar':
        return (
          <div className={`flex items-center gap-4 ${className}`}>
            <div className="skeleton-loader h-16 w-16 rounded-full flex-shrink-0" />
            <div className="flex-1 space-y-2">
              <div className="skeleton-loader h-4 w-32 rounded-lg" />
              <div className="skeleton-loader h-3 w-24 rounded-lg" />
            </div>
          </div>
        )

      default:
        return (
          <div className={`skeleton-loader h-32 w-full rounded-2xl ${className}`} />
        )
    }
  }

  return (
    <div className="space-y-4">
      {skeletons.map((i) => (
        <motion.div
          key={i}
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.3, delay: i * 0.1 }}
        >
          {renderSkeleton()}
        </motion.div>
      ))}
    </div>
  )
}

// Specific skeleton components for different sections
export function ProjectsSkeleton({ count = 6 }: { count?: number }) {
  return (
    <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
      <SkeletonLoader type="project" count={count} />
    </div>
  )
}

export function StatsSkeleton({ count = 4 }: { count?: number }) {
  return (
    <div className="grid grid-cols-2 lg:grid-cols-4 gap-6">
      <SkeletonLoader type="stat" count={count} />
    </div>
  )
}

export function SkillsSkeleton({ count = 6 }: { count?: number }) {
  return (
    <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
      <SkeletonLoader type="skill" count={count} />
    </div>
  )
}

