'use client'

import { TrendingUp } from 'lucide-react'
import { motion } from 'framer-motion'
import { useEffect, useRef, useState } from 'react'
import CountUp from 'react-countup'
import { useInView } from 'framer-motion'

// Individual stat card with animated counter
function StatCard({ stat, index, itemVariants }: { stat: any; index: number; itemVariants: any }) {
  const ref = useRef(null)
  const isInView = useInView(ref, { once: true })
  const [hasAnimated, setHasAnimated] = useState(false)

  useEffect(() => {
    if (isInView && !hasAnimated) {
      setHasAnimated(true)
    }
  }, [isInView, hasAnimated])

  return (
    <motion.div
      ref={ref}
      variants={itemVariants}
      className="relative group"
    >
      <div className="bg-white/80 dark:bg-gray-800/50 backdrop-blur-sm border border-gray-200 dark:border-gray-700 rounded-2xl p-6 hover:border-gray-300 dark:hover:border-gray-600 transition-all duration-300 hover:shadow-glow-duo gpu-accelerated">
        <div className="flex items-start justify-between mb-2">
          <h3 className="text-4xl font-bold text-gradient">
            {hasAnimated ? (
              <CountUp
                start={0}
                end={stat.value}
                duration={2.5}
                decimals={stat.decimals}
                decimal="."
                suffix={stat.suffix}
                useEasing={true}
                easingFn={(t, b, c, d) => {
                  // easeOutExpo easing function
                  return c * (-Math.pow(2, -10 * t / d) + 1) + b
                }}
              />
            ) : (
              <span>0{stat.suffix}</span>
            )}
          </h3>
          <div className="flex items-center gap-1 px-2 py-1 bg-green-500/10 text-green-600 dark:text-green-400 rounded-full text-xs">
            <TrendingUp className="w-3 h-3" />
            {stat.change}
          </div>
        </div>
        <p className="text-gray-800 dark:text-gray-300 font-medium mb-1">{stat.label}</p>
        <p className="text-gray-600 dark:text-gray-500 text-sm">{stat.description}</p>
        
        {/* Hover Effect */}
        <div className="absolute inset-0 bg-gradient-to-r from-red-500/5 to-blue-600/5 rounded-2xl opacity-0 group-hover:opacity-100 transition-opacity duration-300 pointer-events-none"></div>
      </div>
    </motion.div>
  )
}

export function Stats() {
  const stats = [
    {
      value: 50,
      suffix: '+',
      label: 'Projects Completed',
      change: '+12.5%',
      description: 'than last year',
      decimals: 0,
    },
    {
      value: 5.6,
      suffix: 'k+',
      label: 'GitHub Stars',
      change: '+25.4%',
      description: 'Open Source',
      decimals: 1,
    },
    {
      value: 300,
      suffix: 'k+',
      label: 'Lines of Code',
      change: '+8%',
      description: 'Last 30 Days',
      decimals: 0,
    },
    {
      value: 100,
      suffix: '+',
      label: 'Happy Clients',
      change: '+35.8%',
      description: 'This year',
      decimals: 0,
    },
  ]

  const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.1,
      },
    },
  }

  const itemVariants = {
    hidden: { opacity: 0, y: 20 },
    visible: {
      opacity: 1,
      y: 0,
      transition: {
        duration: 0.5,
      },
    },
  }

  return (
    <section className="py-20 bg-gray-50/50 dark:bg-gray-900/50">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          variants={containerVariants}
          initial="hidden"
          whileInView="visible"
          viewport={{ once: true, margin: "-100px" }}
          className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8"
        >
          {stats.map((stat, index) => (
            <StatCard key={index} stat={stat} index={index} itemVariants={itemVariants} />
          ))}
        </motion.div>
      </div>
    </section>
  )
}

