'use client'

import { Carousel } from 'antd'
import { motion } from 'framer-motion'
import { Quote, Star } from 'lucide-react'

interface Testimonial {
  name: string
  role: string
  company: string
  avatar: string
  content: string
  rating: number
}

const testimonials: Testimonial[] = [
  {
    name: 'Marcus Hartwell',
    role: 'CTO',
    company: 'FinTech Innovations',
    avatar: 'https://api.dicebear.com/7.x/avataaars/svg?seed=Marcus',
    content: 'The two-phase approach from Bless & Soul is brilliant! Andrew got our MVP running in record time, then Seed polished it to perfection. It\'s like watching your project get systematically enhanced to maximum power. The gaming analogy isn\'t just branding - it\'s exactly how they work.',
    rating: 5,
  },
  {
    name: 'Sofia Chen',
    role: 'Founder & CEO',
    company: 'CloudNest SaaS',
    avatar: 'https://api.dicebear.com/7.x/avataaars/svg?seed=Sofia',
    content: 'I was skeptical about the "jewel upgrade" concept at first, but it turned out to be genius. We got a working product fast with Andrew, then Seed took it to production-grade quality. Best of both worlds - speed AND perfection. Our investors were incredibly impressed with the final result.',
    rating: 5,
  },
  {
    name: 'James Rodriguez',
    role: 'Product Manager',
    company: 'HealthTech Solutions',
    avatar: 'https://api.dicebear.com/7.x/avataaars/svg?seed=James',
    content: 'Having worked with many dev teams, Bless & Soul\'s phased approach is refreshing. No more endless MVPs that never reach production quality. Andrew builds fast, Seed perfects thoroughly. Clear handoffs, no confusion, stellar results. They\'ve mastered the art of systematic enhancement.',
    rating: 5,
  },
  {
    name: 'Elena Volkov',
    role: 'Head of Engineering',
    company: 'DataStream Analytics',
    avatar: 'https://api.dicebear.com/7.x/avataaars/svg?seed=Elena',
    content: 'The complementary skill sets of Andrew and Seed are remarkable. Andrew\'s rapid prototyping gave us quick validation, then Seed\'s meticulous optimization brought it to enterprise standards. Our application now handles 10x the load we initially anticipated. True professionals who deliver on promises.',
    rating: 5,
  },
  {
    name: 'Ryan Thompson',
    role: 'CEO',
    company: 'StartupLab Ventures',
    avatar: 'https://api.dicebear.com/7.x/avataaars/svg?seed=Ryan',
    content: 'We needed to launch fast but couldn\'t compromise on quality. Bless & Soul delivered both. Andrew got us to market in 3 weeks, Seed made it bulletproof in another 4. The +0 to +15 enhancement isn\'t just marketing - it\'s a proven methodology. Highly recommend for any serious project.',
    rating: 5,
  },
  {
    name: 'Priya Sharma',
    role: 'Founder',
    company: 'EduTech Global',
    avatar: 'https://api.dicebear.com/7.x/avataaars/svg?seed=Priya',
    content: 'Best development experience I\'ve ever had. The Bless phase got our platform running so we could start user testing immediately. The Soul phase then addressed every piece of feedback with production-ready solutions. It\'s rare to find a team that balances speed with such thorough craftsmanship.',
    rating: 5,
  },
]

export function Testimonials() {
  return (
    <section className="py-20 bg-white dark:bg-gray-950">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Header */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5 }}
          className="text-center mb-16"
        >
          <h2 className="text-4xl md:text-5xl font-bold text-gradient mb-4">
            Client Testimonials
          </h2>
          <p className="text-xl text-gray-600 dark:text-gray-400 max-w-3xl mx-auto">
            Don't just take our word for it. Here's what our clients have to say about working with us.
          </p>
        </motion.div>

        {/* Testimonials Carousel */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5, delay: 0.2 }}
          className="max-w-4xl mx-auto testimonials-carousel-wrapper"
        >
          <Carousel
            autoplay
            autoplaySpeed={5000}
            dots={true}
            dotPosition="bottom"
            effect="fade"
          >
            {testimonials.map((testimonial, index) => (
              <div key={index}>
                <div className="px-4 py-8">
                  <div className="bg-white dark:bg-gray-800/50 backdrop-blur-sm border border-gray-200 dark:border-gray-700 rounded-2xl p-8 md:p-12 hover:border-gray-300 dark:hover:border-gray-600 transition-all duration-300 hover:shadow-glow-duo">
                    {/* Quote Icon */}
                    <div className="flex justify-center mb-6">
                      <div className="w-16 h-16 bg-gradient-to-br from-red-500 to-blue-600 rounded-full flex items-center justify-center shadow-glow-duo">
                        <Quote className="w-8 h-8 text-white" />
                      </div>
                    </div>

                    {/* Rating */}
                    <div className="flex justify-center gap-1 mb-6">
                      {Array.from({ length: testimonial.rating }).map((_, i) => (
                        <Star
                          key={i}
                          className="w-5 h-5 fill-yellow-400 text-yellow-400"
                        />
                      ))}
                    </div>

                    {/* Content */}
                    <p className="text-gray-700 dark:text-gray-300 text-lg md:text-xl text-center mb-8 leading-relaxed italic">
                      "{testimonial.content}"
                    </p>

                    {/* Author */}
                    <div className="flex flex-col items-center">
                      <img
                        src={testimonial.avatar}
                        alt={testimonial.name}
                        className="w-16 h-16 rounded-full mb-4 border-2 border-gradient"
                      />
                      <h4 className="text-gray-900 dark:text-white font-bold text-lg">
                        {testimonial.name}
                      </h4>
                      <p className="text-gray-600 dark:text-gray-400 text-sm">
                        {testimonial.role} at {testimonial.company}
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </Carousel>
        </motion.div>

        {/* Stats */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.5, delay: 0.3 }}
          className="mt-16 grid grid-cols-1 sm:grid-cols-3 gap-6 max-w-3xl mx-auto"
        >
          <div className="text-center p-6 bg-gray-100 dark:bg-gray-800/50 backdrop-blur-sm border border-gray-200 dark:border-gray-700 rounded-xl">
            <div className="text-4xl font-bold text-gradient mb-2">100+</div>
            <div className="text-gray-600 dark:text-gray-400">Happy Clients</div>
          </div>
          <div className="text-center p-6 bg-gray-100 dark:bg-gray-800/50 backdrop-blur-sm border border-gray-200 dark:border-gray-700 rounded-xl">
            <div className="text-4xl font-bold text-gradient mb-2">5.0</div>
            <div className="text-gray-600 dark:text-gray-400">Average Rating</div>
          </div>
          <div className="text-center p-6 bg-gray-100 dark:bg-gray-800/50 backdrop-blur-sm border border-gray-200 dark:border-gray-700 rounded-xl">
            <div className="text-4xl font-bold text-gradient mb-2">98%</div>
            <div className="text-gray-600 dark:text-gray-400">Client Satisfaction</div>
          </div>
        </motion.div>
      </div>

      {/* Custom Carousel Dots Styling */}
      <style jsx global>{`
        .testimonials-carousel-wrapper .slick-slider {
          position: relative;
          display: block;
          box-sizing: border-box;
          user-select: none;
        }
        
        .testimonials-carousel-wrapper .slick-list {
          position: relative;
          display: block;
          overflow: hidden;
          margin: 0;
          padding: 0;
        }
        
        .testimonials-carousel-wrapper .slick-track {
          position: relative;
          top: 0;
          left: 0;
          display: flex;
          align-items: flex-start;
        }
        
        .testimonials-carousel-wrapper .slick-slide {
          float: left;
          height: auto;
          min-height: 1px;
          display: block;
        }
        
        .testimonials-carousel-wrapper .slick-dots {
          position: relative;
          bottom: 0;
          display: flex !important;
          flex-direction: row !important;
          justify-content: center;
          align-items: center;
          margin: 24px 0 0 0;
          padding: 0;
          list-style: none;
          text-align: center;
          gap: 8px;
        }
        
        .testimonials-carousel-wrapper .slick-dots li {
          position: relative;
          display: inline-block;
          width: auto;
          height: auto;
          margin: 0;
          padding: 0;
          cursor: pointer;
        }
        
        .testimonials-carousel-wrapper .slick-dots li button {
          display: block;
          width: 12px;
          height: 12px;
          padding: 0;
          border: none;
          border-radius: 50%;
          background: #d1d5db;
          text-indent: -9999px;
          cursor: pointer;
          transition: all 0.3s ease;
        }
        
        .dark .testimonials-carousel-wrapper .slick-dots li button {
          background: #4b5563;
        }
        
        .testimonials-carousel-wrapper .slick-dots li.slick-active button {
          width: 32px;
          border-radius: 6px;
          background: linear-gradient(135deg, #ef4444, #2563eb);
        }
        
        .dark .testimonials-carousel-wrapper .slick-dots li.slick-active button {
          background: linear-gradient(135deg, #ef4444, #2563eb);
        }
        
        .testimonials-carousel-wrapper .slick-dots li:hover button {
          background: #9ca3af;
        }
        
        .dark .testimonials-carousel-wrapper .slick-dots li:hover button {
          background: #6b7280;
        }
      `}</style>
    </section>
  )
}

