'use client'

import React, { useRef, MouseEvent as ReactMouseEvent } from 'react'
import { motion } from 'framer-motion'

interface RippleButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  children: React.ReactNode
  variant?: 'primary' | 'secondary' | 'outline' | 'ghost'
  size?: 'sm' | 'md' | 'lg'
  fullWidth?: boolean
  loading?: boolean
  icon?: React.ReactNode
  iconPosition?: 'left' | 'right'
  as?: 'button' | 'a'
  href?: string
}

export function RippleButton({
  children,
  variant = 'primary',
  size = 'md',
  fullWidth = false,
  loading = false,
  icon,
  iconPosition = 'right',
  className = '',
  onClick,
  disabled,
  as = 'button',
  href,
  ...props
}: RippleButtonProps) {
  const buttonRef = useRef<HTMLButtonElement>(null)

  const createRipple = (event: ReactMouseEvent<HTMLButtonElement>) => {
    const button = buttonRef.current
    if (!button) return

    const circle = document.createElement('span')
    const diameter = Math.max(button.clientWidth, button.clientHeight)
    const radius = diameter / 2

    const rect = button.getBoundingClientRect()
    circle.style.width = circle.style.height = `${diameter}px`
    circle.style.left = `${event.clientX - rect.left - radius}px`
    circle.style.top = `${event.clientY - rect.top - radius}px`
    circle.classList.add('ripple-effect')

    const ripple = button.getElementsByClassName('ripple-effect')[0]
    if (ripple) {
      ripple.remove()
    }

    button.appendChild(circle)

    // Remove the ripple after animation
    setTimeout(() => {
      circle.remove()
    }, 600)
  }

  const handleClick = (event: ReactMouseEvent<HTMLButtonElement>) => {
    if (!disabled && !loading) {
      createRipple(event)
      onClick?.(event)
    }
  }

  const variantClasses = {
    primary: 'bg-gradient-to-r from-red-500 to-blue-600 text-white hover:shadow-glow-duo',
    secondary: 'bg-gray-800 text-white border border-gray-700 hover:border-gray-600 hover:bg-gray-700',
    outline: 'bg-transparent text-white border-2 border-gray-700 hover:border-gray-600 hover:bg-gray-800/50',
    ghost: 'bg-transparent text-gray-300 hover:bg-gray-800/50 hover:text-white',
  }

  const sizeClasses = {
    sm: 'px-4 py-2 text-sm',
    md: 'px-6 py-3 text-base',
    lg: 'px-8 py-4 text-lg',
  }

  const baseClasses = `
    relative overflow-hidden
    font-semibold rounded-full
    transition-all duration-300
    transform hover:scale-105
    disabled:opacity-50 disabled:cursor-not-allowed disabled:hover:scale-100
    focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500
    ${variantClasses[variant]}
    ${sizeClasses[size]}
    ${fullWidth ? 'w-full' : ''}
    ${className}
  `

  const content = (
    <>
      {loading && (
        <span className="absolute inset-0 flex items-center justify-center">
          <svg className="animate-spin h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
          </svg>
        </span>
      )}
      <span className={`flex items-center justify-center gap-2 ${loading ? 'opacity-0' : 'opacity-100'}`}>
        {icon && iconPosition === 'left' && icon}
        {children}
        {icon && iconPosition === 'right' && icon}
      </span>
    </>
  )

  if (as === 'a' && href) {
    return (
      <motion.a
        href={href}
        className={baseClasses}
        whileHover={{ scale: disabled || loading ? 1 : 1.05 }}
        whileTap={{ scale: disabled || loading ? 1 : 0.95 }}
      >
        {content}
      </motion.a>
    )
  }

  return (
    <motion.button
      ref={buttonRef}
      className={baseClasses}
      onClick={handleClick}
      disabled={disabled || loading}
      whileHover={{ scale: disabled || loading ? 1 : 1.05 }}
      whileTap={{ scale: disabled || loading ? 1 : 0.95 }}
      {...props}
    >
      {content}
    </motion.button>
  )
}

// Export a simpler version for common use cases
export function PrimaryButton(props: Omit<RippleButtonProps, 'variant'>) {
  return <RippleButton {...props} variant="primary" />
}

export function SecondaryButton(props: Omit<RippleButtonProps, 'variant'>) {
  return <RippleButton {...props} variant="secondary" />
}

export function OutlineButton(props: Omit<RippleButtonProps, 'variant'>) {
  return <RippleButton {...props} variant="outline" />
}

export function GhostButton(props: Omit<RippleButtonProps, 'variant'>) {
  return <RippleButton {...props} variant="ghost" />
}

